/*---------------------------------------------------------------------------
    Copyright 2008 Andrzej Popowski, www.anpo.republika.pl

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
---------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <stdarg.h>

#include <FL/Fl.H>
#include <FL/Fl_Light_Button.H>
#include <FL/Fl_Check_Button.H>
#include <FL/Fl_Int_Input.H>
#include <FL/Fl_Output.H>
#include <FL/Fl_Text_Display.H>
#include <FL/Fl_File_Input.H>
#include <FL/fl_ask.H>
#include <FL/filename.H>
#include <FL/x.H>
#include <FL/Fl_Preferences.H>

//#define NATIVE_FC

#ifdef NATIVE_FC
#include <FL/Fl_Native_File_Chooser.H>
#else
#include <FL/Fl_File_Chooser.H>
#endif

#include "nutrak.h"

#ifndef NUTRAK_CMD

#define BUFOR_LEN 10000

Fl_Window	    *window;

#ifdef NATIVE_FC
Fl_Native_File_Chooser *fc_inp;
Fl_Native_File_Chooser *fc_out;
#else
Fl_File_Chooser *fc_inp;
Fl_File_Chooser *fc_out;
#endif

Fl_File_Input   *tx_inp;
Fl_File_Input   *tx_out;
Fl_Int_Input    *intime;

Fl_Check_Button	*bindir;
Fl_Check_Button	*bmilisec;
Fl_Check_Button	*bheading;

Fl_Button       *binput;
Fl_Button       *boutput;

Fl_Text_Display *tx_win;

Fl_Text_Buffer  *tx_buf;
Fl_Button       *babout;
Fl_Button       *bconvert;
Fl_Button       *bstop;
Fl_Check_Button	*bovwr;

Fl_Group       *grinp;

int break_work = 0;


void window_position(int ow_x, int ow_y)
{
    int x = 0;
    int y = 0;
    int w = 640;
    int h = 480;
    Fl::screen_xywh(x, y, w, h);

    if (ow_x + window->w() > x + w) ow_x = x + w - window->w();
    if (ow_y + window->h() > y + h) ow_y = y + h - window->h();
    if (ow_x < x) ow_x = x;
    if (ow_y < y) ow_y = y;

    window->position(ow_x, ow_y);
}

void read_pref(void)
{
    char dir_buf[1040];
    int x, y;

    Fl_Preferences pref(Fl_Preferences::USER, "anpo.republika.pl", "wNutrak");

    if (pref.get("InputDir", dir_buf, "", sizeof(dir_buf) -2) != 0)
        tx_inp->value(dir_buf);
    if (pref.get("OutputDir", dir_buf, "", sizeof(dir_buf) -2) != 0)
        tx_out->value(dir_buf);

    pref.get("LogsDir", x, 1);
    bindir->value(x);

    pref.get("PositionX", x, 100);
    pref.get("PositionY", y, 150);
    window_position(x, y);
}

void write_pref(void)
{
    Fl_Preferences pref(Fl_Preferences::USER, "anpo.republika.pl", "wNutrak");
    pref.set("InputDir", tx_inp->value());
    pref.set("OutputDir", tx_out->value());
    pref.set("LogsDir", bindir->value());
    pref.set("PositionX", window->x());
    pref.set("PositionY", window->y());
}

void scroll_tx_win(void)
{
    int len = tx_win->buffer()->length();
    if (len > 2*BUFOR_LEN) {
        int poz = tx_win->buffer()->line_start(len - BUFOR_LEN) -1;
        if (poz > 0) {
            tx_win->buffer()->remove(0, poz);
            len = tx_win->buffer()->length();
        }
    }
    tx_win->insert_position(len);
    tx_win->show_insert_position();
}

void display_tx(const char *fmt, ...)
{
	char bf[1024];
	va_list args;
	
	va_start(args, fmt);
	vsprintf(bf, fmt, args);
	va_end(args);
	
    tx_win->buffer()->append(bf);
    scroll_tx_win();
}


void about_callback(void)
{
    fl_message(
    "Nutrak " N_VERSION " (C) AP\n"
    "www.anpo.republika.pl"
    );
}

void input_callback(void)
{
    #ifdef NATIVE_FC
    if (bindir->value()) {
        fc_inp->type(Fl_Native_File_Chooser::BROWSE_SAVE_DIRECTORY);
        fc_inp->title("Input directory");
        if (tx_inp->size() > 0) {
            fc_inp->directory(tx_inp->value());
        }
        fc_inp->show();
    }
    else {
        fc_inp->type(Fl_Native_File_Chooser::BROWSE_FILE);
        fc_inp->title("Input file");
        if (tx_inp->size() > 0) {
            fc_inp->preset_file(tx_inp->value());
        }
        fc_inp->show();
    }
    if (fc_inp->count() > 0) {
        tx_inp->value(fc_inp->filename());
        tx_inp->redraw();
    }

    #else

    if (bindir->value()) {
        fc_inp->type(Fl_File_Chooser::SINGLE | Fl_File_Chooser::DIRECTORY);
        fc_inp->label("Input directory");
    }
    else {
        fc_inp->type(Fl_File_Chooser::SINGLE);
        fc_inp->label("Input file");
    }
    if (tx_inp->size() > 0) {
        fc_inp->value(tx_inp->value());
    }

    fc_inp->show();

    while (fc_inp->visible()) {
        Fl::wait();
    }

    if (fc_inp->count() > 0) {
        tx_inp->value(fc_inp->value());
        tx_inp->redraw();
    }
    #endif
}

void output_callback(void)
{
    #ifdef NATIVE_FC
    if (tx_out->size() > 0) {
        fc_out->directory(tx_out->value());
    }
    fc_out->show();
    
    if (fc_out->count() > 0) {
        tx_out->value(fc_out->filename());
        tx_out->redraw();
    }

    #else

    if (tx_out->size() > 0) {
        fc_out->value(tx_out->value());
    }

    fc_out->show();

    while (fc_out->visible()) {
        Fl::wait();
    }

    if (fc_out->count() > 0) {
        tx_out->value(fc_out->value());
        tx_out->redraw();
    }
    #endif
}

int cat_fname(char *d, int il, const char *fn, int ln, int max)
{
    if (il + ln + 2 > max)
        return 0;

    memcpy(d + il, fn, ln);
    d[il + ln] = 0;

    return 1;
}

int dir_scan(const char *inp_dir, int lev, const char *out_dir, unsigned opt, int t)
{
//    display_tx("\nscan input %s\n", inp_dir);

    char   dbuf[1040];
    dirent **list = NULL;
    int    lnum   = 0;
    int    ncw    = 0;

    int iln = strlen(inp_dir);
    if (iln > 1000)
        return 0;
    memcpy(dbuf, inp_dir, iln);
    char lc = (iln > 0)? dbuf[iln -1]: 0;
    if (lc != ':' && lc != '/' && lc != '\\') {
        dbuf[iln] = '/';
        iln++;
    }

    lnum = fl_filename_list(inp_dir, &list);

    for (int i = 0; i < lnum; i++) {
        if (break_work)
            break;

        char *de = list[i]->d_name;
        int   ln = strlen(de);
        if (ln <= 0)
            continue;

//        display_tx("scan dirent %s\n", de);
  
        if (de[ln -1] == '/') {
            if (lev != 0)
                continue;
            if (strcmp(list[i]->d_name, "./") == 0)
                continue;
            if (strcmp(list[i]->d_name, "../") == 0)
                continue;
            if (!cat_fname(dbuf, iln, de, ln, sizeof(dbuf)))
                continue;
            ncw += dir_scan(dbuf, 1, out_dir, opt, t);
            continue;
        }
        if (fl_filename_match(list[i]->d_name, "gps.bin")) {
            if (!cat_fname(dbuf, iln, de, ln, sizeof(dbuf)))
                continue;
            if (nutrak(dbuf, out_dir, opt, t) == 0)
                ncw++;
            continue;
        }
    }

    for (int i = lnum; i > 0;) {
        free((void*)(list[--i]));
    }
    free((void*)list);

    return ncw;
}

void work_begin(void)
{
    break_work = 0;

    binput->deactivate();
    bindir->deactivate();
    babout->deactivate();
    tx_inp->deactivate();
    boutput->deactivate();
    bmilisec->deactivate();
    bheading->deactivate();
    intime->deactivate();
    tx_out->deactivate();
    bovwr->deactivate();

    bconvert->deactivate();
    bconvert->hide();
    bstop->activate();
    bstop->show();
}


void work_end(void)
{
    if (break_work) {
        display_tx("User break!\n");
        break_work = 0;
    }

    bstop->deactivate();
    bstop->hide();
    bconvert->activate();
    bconvert->show();

    binput->activate();
    bindir->activate();
    babout->activate();
    tx_inp->activate();
    boutput->activate();
    bmilisec->activate();
    bheading->activate();
    intime->activate();
    tx_out->activate();
    bovwr->activate();
}


void stop_callback(void)
{
    break_work = 1;
}

int test_break(void)
{
    Fl::check();
    return break_work;
}

void convert_callback(void)
{
    char buf[200];
    int  tmoff;

    if (intime->size() > 0) {
        tmoff = strtol(intime->value(), NULL, 10);

        sprintf(buf, "%d", tmoff);
        intime->value(buf);
    }
    else {
        tmoff = 0;
        intime->value("");
    }
    intime->redraw();

    if (strlen(tx_inp->value()) <= 0) {
        if (bindir->value())
            display_tx("Input directory missing!\n");
        else 
            display_tx("Input file missing!\n");
        return;
    }
    if (bindir->value() && !fl_filename_isdir(tx_inp->value())) {
        display_tx("Input directory not valid!\n");
        return;
    }
    if (strlen(tx_out->value()) <= 0) {
        display_tx("Output directory missing!\n");
        return;
    }
    if (bindir->value() && !fl_filename_isdir(tx_out->value())) {
        display_tx("Output directory not valid!\n");
        return;
    }

    display_tx("\nConvert:\n");
    if (bindir->value()) {
        display_tx("Input directory:  %s\n", tx_inp->value());
    }
    else {
        display_tx("Input file name:  %s\n", tx_inp->value());
    }

    display_tx("Output directory:  %s\n", tx_out->value());

    if (bmilisec->value() != 0 || bheading->value() != 0 || tmoff != 0) {
        display_tx("Options: ");
        int coma = 0;
        if (bmilisec->value()) {
            display_tx(" Miliseconds");
            coma = 1;
        }
        if (bheading->value()) {
            if (coma) display_tx(",");
            display_tx(" Heading");
            coma = 1;
        }
        if (tmoff) {
            if (coma) display_tx(",");
            sprintf(buf, " Time %+d", tmoff);
            display_tx(buf);
        }
        display_tx(".\n");
    }
    if (bovwr->value()) {
        display_tx("Overwrite files.\n");
    }
    

    display_tx("\n");
    unsigned opt = (bmilisec->value()? OPT_MILISEC: 0)
                | (bheading->value()? OPT_COURSE: 0) 
                | (bovwr->value()? OPT_OVWR: 0);

    work_begin();
    Fl::check();
    if (!bindir->value()) {
        if (nutrak(tx_inp->value(), tx_out->value(), opt, tmoff) == 0) {
            display_tx("\nOK\n");
        }
        work_end();
        return;
    }

    int ncw = dir_scan(tx_inp->value(), 0, tx_out->value(), opt, tmoff);
    display_tx("\nConverted %d files\n", ncw);
    work_end();
}

static void window_cb(void)
{
    break_work = 1;
    window->hide();
}


int main(int  argc,	char *argv[])
{
    Fl::scheme("gtk+");
    
    window = new Fl_Window(500, 400, "wNutrak");
    window->callback((Fl_Callback *)window_cb);
    #ifdef __WIN32
    window->icon((char *)LoadIcon(fl_display, MAKEINTRESOURCE(501)));
    #endif

    babout = new Fl_Button(475, 15, 15, 15);
    babout->callback((Fl_Callback *)about_callback);
    babout->type(0);

    // button for input
    binput = new Fl_Button(10, 10, 80, 25, "Input");
    binput->callback((Fl_Callback *)input_callback);
    tx_inp = new Fl_File_Input(10, 45, 480, 35);
    
    bindir = new Fl_Check_Button(130, 10, 120, 25, "Logs directory");

    // button for output
    boutput = new Fl_Button(10, 100, 80, 25, "Output");
    boutput->callback((Fl_Callback *)output_callback);
    tx_out = new Fl_File_Input(10, 135, 480, 35);

    bmilisec = new Fl_Check_Button(130, 100, 100, 25, "Miliseconds");
    bheading = new Fl_Check_Button(260, 100, 80, 25, "Heading");
    intime = new Fl_Int_Input(410, 100, 80, 25, "Time ");
    
    // output window
    tx_win = new Fl_Text_Display(10, 185, 480, 165);
    tx_buf = new Fl_Text_Buffer(BUFOR_LEN);
    tx_win->buffer(tx_buf);

    // action
    bconvert = new Fl_Button(10, 365, 80, 25, "Convert");
    bconvert->callback((Fl_Callback *)convert_callback);

    bstop    = new Fl_Button(10, 365, 80, 25, "Stop");
    bstop->callback((Fl_Callback *)stop_callback);
    bstop->deactivate();
    bstop->hide();

    bovwr = new Fl_Check_Button(130, 365, 120, 25, "Overwrite files");

    // dialogs
    #ifdef NATIVE_FC
    fc_inp = new Fl_Native_File_Chooser();
    fc_out = new Fl_Native_File_Chooser(Fl_Native_File_Chooser::BROWSE_SAVE_DIRECTORY);
    #else
    fc_inp = new Fl_File_Chooser("", "*\t*.bin",
        Fl_File_Chooser::SINGLE,
        "Input directory");
    fc_inp->preview(0);
    fc_inp->previewButton->hide();
    fc_inp->sort = fl_casealphasort;

    fc_out = new Fl_File_Chooser("", "*",
        Fl_File_Chooser::SINGLE | Fl_File_Chooser::DIRECTORY | Fl_File_Chooser::CREATE,
        "Output directory");
    fc_out->preview(0);
    fc_out->previewButton->hide();
    fc_out->sort = fl_casealphasort;
    #endif
    
    window->size_range(400, 400);
    window->resizable(tx_win);
    window->end();

    read_pref();

    OPTIONS_STU op;
    parse_arg(argc, argv, &op);
    if ((op.options & OPT_ERROR) == 0) {
        if (op.options & OPT_MILISEC) {
            bmilisec->value(1);
        }
        if (op.options & OPT_COURSE) {
            bheading->value(1);
        }
        if (op.options & OPT_TIME) {
            char bf[40];
            sprintf(bf, "%d", op.tmoff);
            intime->value(bf);
        }
        if (op.options & OPT_SOURCE) {
            tx_inp->value(op.infile);
            bindir->value(fl_filename_isdir(op.infile)? 1: 0);
        }
        if (op.options & OPT_DESTINATION) {
            tx_out->value(op.outdir);
        }
        if (op.options & OPT_OVWR) {
            bovwr->value(1);
        }
    }

    display_tx("\n");
    display_tx("Provide input data and press [Convert].\n");
    display_tx("\n");
    display_tx("You can provide nuvi logs directory as an input,\n");
    display_tx("wNutrak will convert all files gps.bin from logs subdirectories.\n");
    display_tx("\n");

    window->show(1, argv);

    Fl::run();

    write_pref();
    delete window;
    return (0);
}

#endif // !NUTRAK_CMD
