/*---------------------------------------------------------------------------
    Copyright 2008 Andrzej Popowski, www.anpo.republika.pl

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
---------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <unistd.h>

//#define NUTRAK_CMD

#include "nutrak.h"



using namespace std;

struct __attribute__ ((packed)) HEADER  {
       unsigned     start;
       unsigned     week;
       unsigned     second;
       float        course;
       double       longitude;
       double       latitude;
       char         empty[12];
       unsigned     num;
};

struct __attribute__ ((packed)) RECORD {
       unsigned     start;
       unsigned     msec;
       double       latitude;
       double       longitude;
       float        course;       
};

unsigned char xor_key[243] = {
0x61, 0x8F, 0x54, 0x5B, 0x77, 0xE7, 0xC1, 0x29,
0xA2, 0xA0, 0x98, 0x78, 0xEB, 0xD1, 0x69, 0xAF,
0xD4, 0x75, 0xDF, 0xA1, 0x9C, 0x88, 0x38, 0xDE,
0x9D, 0x8C, 0x48, 0x2B, 0xAA, 0xC0, 0x25, 0x92,
0x60, 0x8B, 0x44, 0x1B, 0x6A, 0xB3, 0xE4, 0xB5,
0xEC, 0xD5, 0x79, 0xEF, 0xE1, 0xA9, 0xBC, 0x15,
0x52, 0x53, 0x57, 0x67, 0xA7, 0xB4, 0xE8, 0xC5,
0x39, 0xE2, 0xAD, 0xCC, 0x55, 0x5F, 0x87, 0x34,
0xCE, 0x5D, 0x7F, 0x14, 0x4E, 0x43, 0x17, 0x5A,
0x73, 0xD7, 0x81, 0x1C, 0x6E, 0xC3, 0x31, 0xC2,
0x2D, 0xB2, 0xE0, 0xA5, 0xAC, 0xC8, 0x45, 0x1F,
0x7A, 0x00, 0xF1, 0xE9, 0xC9, 0x49, 0x2F, 0xBA,
0x0D, 0x32, 0xC6, 0x3D, 0xF2, 0xED, 0xD9, 0x89,
0x3C, 0xEE, 0xDD, 0x99, 0x7C, 0x08, 0x1E, 0x76,
0xE3, 0xB1, 0xDC, 0x95, 0x6C, 0xBB, 0x11, 0x42,
0x13, 0x4A, 0x33, 0xCA, 0x4D, 0x3F, 0x07, 0x1A,
0x66, 0xA3, 0xA4, 0xA8, 0xB8, 0x05, 0x12, 0x46,
0x23, 0x8A, 0x40, 0x0B, 0x2A, 0xA6, 0xB0, 0xD8,
0x85, 0x2C, 0xAE, 0xD0, 0x65, 0x9F, 0x94, 0x68,
0xAB, 0xC4, 0x35, 0xD2, 0x6D, 0xBF, 0x21, 0x82,
0x20, 0x7E, 0x10, 0x3E, 0x03, 0x0A, 0x26, 0x96,
0x70, 0xCB, 0x51, 0x4F, 0x47, 0x27, 0x9A, 0x80,
0x18, 0x5E, 0x83, 0x24, 0x8E, 0x50, 0x4B, 0x37,
0xDA, 0x8D, 0x4C, 0x3B, 0xEA, 0xCD, 0x59, 0x6F,
0xC7, 0x41, 0x0F, 0x3A, 0xE6, 0xBD, 0x19, 0x62,
0x93, 0x64, 0x9B, 0x84, 0x28, 0x9E, 0x90, 0x58,
0x6B, 0xB7, 0x01, 0x02, 0x06, 0x16, 0x56, 0x63,
0x97, 0x74, 0xDB, 0x91, 0x5C, 0x7B, 0x04, 0x0E,
0x36, 0xD6, 0x7D, 0x0C, 0x2E, 0xB6, 0xF0, 0xE5,
0xB9, 0x09, 0x22, 0x86, 0x30, 0xBE, 0x1D, 0x72,
0xD3, 0x71, 0xCF//, 0x61, 0x8F, 0x54, 0x5B, 0x77,
//0xE7, 0xC1, 0x29, 0xA2, 0xA0, 0x98, 0x78, 0xEB
};

unsigned char start[256];

void write_start_gpx_track(FILE *fo)
{
    // wersja GPSBABEL
    fprintf(fo, "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n");
    fprintf(fo, "<gpx version=\"1.0\"");
    fprintf(fo, " creator=\"nutrak " N_VERSION " (C) A.P. www.anpo.republika.pl\"");
    fprintf(fo, " xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"");
    fprintf(fo, " xmlns=\"http://www.topografix.com/GPX/1/0\"");
    fprintf(fo, " xsi:schemaLocation=\"http://www.topografix.com/GPX/1/0 http://www.topografix.com/GPX/1/0/gpx.xsd\"");
    fprintf(fo, ">\n\n");

    fprintf(fo, " <trk>\n");
    fprintf(fo, "  <name>nuvi</name>\n");
    fprintf(fo, "  <trkseg>\n");

}

void write_end_gpx_track(FILE *fo)
{
    fprintf(fo, "  </trkseg>\n");
    fprintf(fo, " </trk>\n");
    fprintf(fo, "</gpx>\n");
}

int write_point(FILE *fo, time_t *gmt, unsigned start, RECORD *rek, int opt)
{
    double xc = 180.0/M_PI;
    fprintf(fo, "   <trkpt lat=\"%3.8lf\" lon=\"%3.8lf\">\n", xc*rek->latitude, xc*rek->longitude);

    if (opt & OPT_COURSE) {
        double course = xc*rek->course;
        if (course < 0.0)
            course = course + 360.0;
        if (course > 360.0)
            course = 360.0;
        fprintf(fo, "    <course>%3.2lf</course>\n", course);
    }

    unsigned  dtim = (rek->msec - start)/1000;
    unsigned  msec = (rek->msec - start)%1000;
    time_t    amt  = *gmt + dtim;
    struct tm *ndt;

    if (opt & OPT_MILISEC) {
        ndt = gmtime(&amt);
        fprintf(fo, "    <time>%04u-%02u-%02uT%02u:%02u:%02u.%03uZ</time>\n",
            ndt->tm_year +1900,
            ndt->tm_mon +1,
            ndt->tm_mday,
            ndt->tm_hour,
            ndt->tm_min,
            ndt->tm_sec,
            msec);
    }
    else {  // bez milisekund
        amt += (msec + 500)/1000;
        ndt  = gmtime(&amt);
        fprintf(fo, "    <time>%04u-%02u-%02uT%02u:%02u:%02uZ</time>\n",
            ndt->tm_year +1900,
            ndt->tm_mon +1,
            ndt->tm_mday,
            ndt->tm_hour,
            ndt->tm_min,
            ndt->tm_sec);
    }

    fprintf(fo, "   </trkpt>\n");
    return 1;
}

int make_file_name(time_t *gmt, const char *p, char *fn, int flen)
{
    int len = 0;
    if (p == NULL || *p == 0) {
        p = getcwd(fn, flen);
        if (p == NULL) {
            return 0;
        }
        len = strlen(p);
        if (len + 22 > flen)
            return 0;
    }
    else {
        len = strlen(p);
        if (len + 22 > flen)
            return 0;
        memcpy(fn, p, len +1);
    }

    char *s  = "";
    #ifdef __WIN32
    if (len > 0 && p[len -1] != ':' && p[len -1] != '\\' && p[len -1] != '/')
        s = "\\";
    #else
    if (len > 0 && p[len -1] != '/')
        s = "/";
    #endif

    struct tm  *ndt = gmtime(gmt);
    sprintf(fn + len, "%s%04u%02u%02u_%02u%02u%02u.gpx",
        s,
        ndt->tm_year +1900,
        ndt->tm_mon +1,
        ndt->tm_mday,
        ndt->tm_hour,
        ndt->tm_min,
        ndt->tm_sec);

    return 1;
}


int decode(unsigned char *d, unsigned len)
{
//  display_tx("decode %02X longitude %u\n", d[0], len);
    unsigned poz = start[unsigned(d[0])];
    if (poz >= 243)
        return 0;

    for (unsigned x = 0; x <= len; x++) {
        d[x] ^= xor_key[poz];
        poz   = (poz +1)%243;
    }
    return 1;
}

void init_decode(void)
{
    memset(start, 0xFF, sizeof(start));
    for (unsigned p = 0; p < 243; p++) {
        start[unsigned(xor_key[p])] = p;
    }
}


int nutrak(const char *infile, const char *path, unsigned opt, int tmoff)
{
//    display_tx("input file %s\n", infile);
//    display_tx("path %s\n", path);

    FILE *fi = fopen(infile, "rb");
    if (fi == NULL) {
        display_tx("input file open error: %s\n", infile);
        return 2;
    }

    init_decode();

    unsigned char buf[100];
    HEADER   nag;
    RECORD   *rek;
    unsigned rpoz = 0;

    if (fread(buf, 1, sizeof(HEADER) +1, fi) != sizeof(HEADER) +1) {
        display_tx("header read error\n");
        return 3;
    }
    memcpy(&nag, buf, sizeof(HEADER));

    if (nag.start != *(unsigned *)"BEGN") {
        if (!decode(buf, sizeof(HEADER))) {
            display_tx("wrong file format\n");
            return 3;
        }
        memcpy(&nag, buf +1, sizeof(HEADER));
        rpoz = 1;
    }

    if (nag.start != *(unsigned *)"BEGN") {
        display_tx("wrong header format\n");
        return 3;
    }

    if (fseek(fi, sizeof(HEADER) + rpoz, SEEK_SET)) {
        display_tx("file seek error: %s\n", infile);
        return 3;
    }

    if (fread(buf, 1, sizeof(RECORD) + rpoz, fi) != sizeof(RECORD) + rpoz) {
        display_tx("first record read error\n");
        return 3;
    }

    if (rpoz)
        decode(buf, sizeof(RECORD));
    rek = (RECORD *)(buf + rpoz);

    if (rek->start != *(unsigned *)"$$$$") {
        display_tx("wrong record format\n");
        return 3;
    }

    tzset();
//  printf("timezone %i, daylight %i, tzname %s %s\n", timezone, daylight, tzname[0], tzname[1]);
       
    struct tm dt;
    dt.tm_sec    = 0;
    dt.tm_min    = 0;
    dt.tm_hour   = 0;
    dt.tm_mday   = 22;
    dt.tm_mon    = 7;
    dt.tm_year   = 99;
    dt.tm_wday   = 0;
    dt.tm_yday   = 0;
    dt.tm_isdst  = 0;

    time_t gmt = mktime(&dt);

    gmt += 7*24*60*60*nag.week + nag.second - timezone + tmoff;
    unsigned start = rek->msec;

    char fn_name[1040];
    if (!make_file_name(&gmt, path, fn_name, sizeof(fn_name))) {
        display_tx("output file name error\n");
        return 2;
    }
    FILE *fo = fopen(fn_name, "wb");
    if (fo == NULL) {
        display_tx("output file open error: %s\n", fn_name);
        return 2;
    }
    #ifdef NUTRAK_CMD
    display_tx("%s -> %s\n", infile, fn_name);
    #else
    display_tx("%s <- %s\n", fn_name, infile);
    #endif

    write_start_gpx_track(fo);
    write_point(fo, &gmt, start, rek, opt);

    #ifdef NUTRAK_CMD
    for (;;) {
    #else
    for (int rx = 0;; rx++) {
        if ((rx % 1000) == 0) {
            if (test_break())
                break;
        }
    #endif
        if (fread(buf, 1, sizeof(RECORD) + rpoz, fi) != sizeof(RECORD) + rpoz) {
            if (!feof(fi)) {
                display_tx("record read error\n");
            }
            break;
        }

        if (rpoz)
            decode(buf, sizeof(RECORD));
        rek = (RECORD *)(buf + rpoz);

        if (rek->start != *(unsigned *)"$$$$") {
            display_tx("wrong record format\n");
            break;
        }
        write_point(fo, &gmt, start, rek, opt);
    }
    write_end_gpx_track(fo);

    fclose(fi);
    fclose(fo);

    return 0;
}


void info(void)
{
    #ifdef NUTRAK_CMD
    display_tx("nutrak v" N_VERSION " (C) A.P. www.anpo.republika.pl\n\n");

    display_tx("use: nutrak [-c] [-m] [-t seconds] [-p path] gps.bin\n");
    #else
    display_tx("\n");
    display_tx("use: wNutrak [-c] [-m] [-t seconds] [-p path] [input_file]\n");
    #endif
    display_tx("\t-c - write course\n");
    display_tx("\t-m - write miliseconds\n");
    display_tx("\t-t - add time offset in seconds\n");
    display_tx("\t-p - output path\n");
}

int parse_arg(int argc, char **argv, OPTIONS_STU *op)
{
    op->options = 0;
    op->infile  = NULL;
    op->outdir  = NULL;
    op->tmoff   = 0;

    if (argc <= 1)
        return 1;

    int      farg = 1;
    char     dummy;

    for (; farg < argc; farg++) {
        char *tar = argv[farg];
        if (tar[0] != '-')
            break;
        if (tar[1] == 0) {
            op->options |= OPT_ERROR;
            display_tx("unknown option %s\n", tar);
            info();
            return 1;
        }
        for (int i = 1; tar[i] != 0; i++) {
            switch (tar[i]) {
                case 'c':
                case 'C':
                    op->options |= OPT_COURSE;
                    break;

                case 'm':
                case 'M':
                    op->options |= OPT_MILISEC;
                    break;

                case 't':
                case 'T':
                    if (tar[i +1] != 0 || farg +1 >= argc) {
                        op->options |= OPT_ERROR;
                        display_tx("missing time offset\n");
                        info();
                        return 1;
                    }
                    farg += 1;
                    if (sscanf(argv[farg],"%d%c", &(op->tmoff), &dummy) != 1) {
                        op->options |= OPT_ERROR;
                        display_tx("wrong time offset %s\n", argv[farg]);
                        info();
                        return 1;
                    }
                    op->options |= OPT_TIME;
                    break;

                case 'p':
                case 'P':
                    if (tar[i +1] != 0 || farg +1 >= argc) {
                        op->options |= OPT_ERROR;
                        display_tx("missing path\n");
                        info();
                        return 1;
                    }
                    farg += 1;
                    op->outdir   = argv[farg];
                    op->options |= OPT_DESTINATION;
                    break;

                default:
                    op->options |= OPT_ERROR;
                    display_tx("unknown option %s\n", tar);
                    info();
                    return 1;
            }
        }
    }

    if (farg >= argc) {
        op->options |= OPT_ERROR;
        info();
        return 1;
    }
    op->infile   = argv[farg];
    op->options |= OPT_SOURCE;

    return 0;
}




#ifdef NUTRAK_CMD

int main(int argc, char *argv[])
{
    if (argc < 2) {
        info();
        return 1;
    }

    OPTIONS_STU op;
    int result = parse_arg(argc, argv, &op);
    if (result)
        return result;

    return nutrak(op.infile, op.outdir, op.options, op.tmoff);
}

#endif //NUTRAK_COMANDLINE
