/*---------------------------------------------------------------------------
    Copyright 2008 Andrzej Popowski, www.anpo.republika.pl

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
---------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <stdarg.h>

#include <FL/Fl_File_Chooser.H>
#include <FL/Fl_Light_Button.H>
#include <FL/Fl_Int_Input.H>
#include <FL/Fl_Output.H>
#include <FL/Fl_Text_Display.H>
#include <FL/fl_ask.H>
#include <FL/filename.H>
#include <FL/x.H>

#include "nutrak.h"

#define BUFOR_LEN 10000

Fl_File_Chooser *fc_inp;
Fl_File_Chooser *fc_out;

int indir     = 1;
int milisec   = 0;
int heading   = 0;
int tmoff     = 0;

Fl_File_Input   *tx_inp;
Fl_File_Input   *tx_out;
Fl_Int_Input    *intime;

Fl_Check_Button	*bindir;
Fl_Check_Button	*bmilisec;
Fl_Check_Button	*bheading;

Fl_Button       *binput;
Fl_Button       *boutput;

Fl_Text_Display *tx_win;

Fl_Text_Buffer *tx_buf;
Fl_Button      *babout;
Fl_Button      *bconvert;
Fl_Button      *bstop;

Fl_Group       *grinp;

int break_work = 0;

void scroll_tx_win(void)
{
    int len = tx_win->buffer()->length();
    if (len > 2*BUFOR_LEN) {
        int poz = tx_win->buffer()->line_start(len - BUFOR_LEN) -1;
        if (poz > 0) {
            tx_win->buffer()->remove(0, poz);
            len = tx_win->buffer()->length();
        }
    }
    tx_win->insert_position(len);
    tx_win->show_insert_position();
}

void display_tx(const char *fmt, ...)
{
	char bf[1024];
	va_list args;
	
	va_start(args, fmt);
	vsprintf(bf, fmt, args);
	va_end(args);
	
    tx_win->buffer()->append(bf);
    scroll_tx_win();
}


void about_callback(void)
{
    fl_message(
    "Nutrak " N_VERSION " (C) AP\n"
    "www.anpo.republika.pl"
    );
}

void input_callback(void)
{
    if (indir) {
        fc_inp->type(Fl_File_Chooser::SINGLE | Fl_File_Chooser::DIRECTORY);
        fc_inp->label("Input directory");
    }
    else {
        fc_inp->type(Fl_File_Chooser::SINGLE);
        fc_inp->label("Input file");
    }
    if (tx_inp->size() > 0)
        fc_inp->value(tx_inp->value());

    fc_inp->show();

    while (fc_inp->visible()) {
        Fl::wait();
    }

    if (fc_inp->count() > 0) {
        tx_inp->value(fc_inp->value());
        tx_inp->redraw();
    }
}

void indir_callback(Fl_Button	*button)
{
    indir = indir? 0: 1;
    button->value(indir);
}

void output_callback(void)
{
    if (tx_out->size() > 0)
        fc_out->value(tx_out->value());

    fc_out->show();

    while (fc_out->visible()) {
        Fl::wait();
    }

    if (fc_out->count() > 0) {
        tx_out->value(fc_out->value());
        tx_out->redraw();
    }
}

void milisec_callback(Fl_Button	*button)
{
    milisec = milisec? 0: 1;
    button->value(milisec);
}

void heading_callback(Fl_Button	*button)
{
    heading = heading? 0: 1;
    button->value(heading);
}

int cat_fname(char *d, int il, const char *fn, int ln, int max)
{
    if (il + ln + 2 > max)
        return 0;

    memcpy(d + il, fn, ln);
    d[il + ln] = 0;

    return 1;
}

int dir_scan(const char *inp_dir, int lev, const char *out_dir, unsigned opt, int t)
{
//    display_tx("\nscan input %s\n", inp_dir);

    char   dbuf[1040];
    dirent **list = NULL;
    int    lnum   = 0;
    int    ncw    = 0;

    int iln = strlen(inp_dir);
    if (iln > 1000)
        return 0;
    memcpy(dbuf, inp_dir, iln);
    char lc = (iln > 0)? dbuf[iln -1]: 0;
    if (lc != ':' && lc != '/' && lc != '\\') {
        dbuf[iln] = '/';
        iln++;
    }

    lnum = fl_filename_list(inp_dir, &list);

    for (int i = 0; i < lnum; i++) {
        char *de = list[i]->d_name;
        int   ln = strlen(de);
        if (ln <= 0)
            continue;

//        display_tx("scan dirent %s\n", de);
  
        if (de[ln -1] == '/') {
            if (lev != 0)
                continue;
            if (strcmp(list[i]->d_name, "./") == 0)
                continue;
            if (strcmp(list[i]->d_name, "../") == 0)
                continue;
            if (!cat_fname(dbuf, iln, de, ln, sizeof(dbuf)))
                continue;
            ncw += dir_scan(dbuf, 1, out_dir, opt, t);
            continue;
        }
        if (fl_filename_match(list[i]->d_name, "gps.bin")) {
            if (!cat_fname(dbuf, iln, de, ln, sizeof(dbuf)))
                continue;
            if (nutrak(dbuf, out_dir, opt, t) == 0)
                ncw++;
            continue;
        }
    }

    for (int i = lnum; i > 0;) {
        free((void*)(list[--i]));
    }
    free((void*)list);

    return ncw;
}

void work_begin(void)
{
    break_work = 0;

    grinp->deactivate();

    bconvert->deactivate();
    bconvert->hide();
    bstop->activate();
    bstop->show();
}


void work_end(void)
{
    if (break_work) {
        display_tx("User break!\n");
        break_work = 0;
    }

    bstop->deactivate();
    bstop->hide();
    bconvert->activate();
    bconvert->show();

    grinp->activate();
}


void stop_callback(void)
{
    break_work = 1;
}

int test_break(void)
{
    Fl::check();
    return break_work;
}

void convert_callback(void)
{
    char buf[200];
    if (intime->size() > 0) {
        tmoff = strtol(intime->value(), NULL, 10);

        sprintf(buf, "%d", tmoff);
        intime->value(buf);
    }
    else {
        tmoff = 0;
        intime->value("");
    }
    intime->redraw();

    if (strlen(tx_inp->value()) <= 0) {
        if (indir)
            tx_win->insert("Input directory missing!\n");
        else 
            display_tx("Input file missing!\n");
        return;
    }
    if (indir && !fl_filename_isdir(tx_inp->value())) {
        display_tx("Input directory not valid!\n");
        return;
    }
    if (strlen(tx_out->value()) <= 0) {
        display_tx("Output directory missing!\n");
        return;
    }
    if (indir && !fl_filename_isdir(tx_out->value())) {
        display_tx("Output directory not valid!\n");
        return;
    }

    display_tx("\nConvert:\n");
    if (indir) {
        display_tx("Input directory:  %s\n", tx_inp->value());
    }
    else {
        display_tx("Input file:       %s\n", tx_inp->value());
    }

    display_tx("Output directory: %s\n", tx_out->value());

    if (milisec != 0 || heading != 0 || tmoff != 0) {
        display_tx("Options:");
        int coma = 0;
        if (milisec) {
            display_tx(" Miliseconds");
            coma = 1;
        }
        if (heading) {
            if (coma) display_tx(",");
            display_tx(" Heading");
            coma = 1;
        }
        if (tmoff) {
            if (coma) display_tx(",");
            sprintf(buf, " Time %+d", tmoff);
            display_tx(buf);
        }
        display_tx("\n");
    }

    display_tx("\n");
    unsigned opt = (milisec? OPT_MILISEC: 0) | (heading? OPT_COURSE: 0);

    work_begin();
    Fl::check();
    if (!indir) {
        if (nutrak(tx_inp->value(), tx_out->value(), opt, tmoff) == 0) {
            display_tx("\nOK\n");
        }
        work_end();
        return;
    }

    int ncw = dir_scan(tx_inp->value(), 0, tx_out->value(), opt, tmoff);
    display_tx("\nConverted %d files\n", ncw);
    work_end();
}


int main(int  argc,	char *argv[])
{
    Fl_Window	   *window;
    
    Fl::scheme("gtk+");
    
    window = new Fl_Window(500, 400, "wNutrak");
    //window->icon((char *)LoadIcon(fl_display, MAKEINTRESOURCE(101 /*IDC_ICON*/)));

    babout = new Fl_Button(475, 15, 15, 15);
    babout->callback((Fl_Callback *)about_callback);
    babout->type(0);

    grinp  = new Fl_Group(5, 5, 490, 175);

    // button for input
    binput = new Fl_Button(10, 10, 80, 25, "Input");
    binput->callback((Fl_Callback *)input_callback);
    
    tx_inp = new Fl_File_Input(10, 45, 480, 35);
    
    bindir = new Fl_Check_Button(130, 10, 120, 25, "Logs directory");
    bindir->callback((Fl_Callback *)indir_callback);
    bindir->type(0);
    bindir->value(indir);
    bindir->box(FL_NO_BOX);

    // button for output
    boutput = new Fl_Button(10, 100, 80, 25, "Output");
    boutput->callback((Fl_Callback *)output_callback);
    
    tx_out = new Fl_File_Input(10, 135, 480, 35);

    bmilisec = new Fl_Check_Button(130, 100, 100, 25, "Miliseconds");
    bmilisec->callback((Fl_Callback *)milisec_callback);
    bmilisec->type(0);
    bmilisec->value(milisec);
    bmilisec->box(FL_NO_BOX);
    
    bheading = new Fl_Check_Button(260, 100, 80, 25, "Heading");
    bheading->callback((Fl_Callback *)heading_callback);
    bheading->type(0);
    bheading->value(heading);
    bheading->box(FL_NO_BOX);

    intime = new Fl_Int_Input(410, 100, 80, 25, "Time ");
    
    grinp->end();

    // output window
    tx_win = new Fl_Text_Display(10, 185, 480, 165);
    tx_buf = new Fl_Text_Buffer(BUFOR_LEN);
    tx_win->buffer(tx_buf);

    // action
    bconvert = new Fl_Button(10, 365, 80, 25, "Convert");
    bconvert->callback((Fl_Callback *)convert_callback);

    bstop    = new Fl_Button(10, 365, 80, 25, "Stop");
    bstop->callback((Fl_Callback *)stop_callback);
    bstop->deactivate();
    bstop->hide();

    // dialogs
    fc_inp = new Fl_File_Chooser(".", "*",
        Fl_File_Chooser::SINGLE,
        "Input directory");
    fc_inp->preview(0);
    fc_inp->sort = fl_casealphasort;

    fc_out = new Fl_File_Chooser(".", "*",
        Fl_File_Chooser::SINGLE | Fl_File_Chooser::DIRECTORY | Fl_File_Chooser::CREATE,
        "Output directory");
    fc_out->preview(0);
    fc_out->sort = fl_casealphasort;
    
    window->size_range(400, 400);
    window->resizable(tx_win);
    window->end();

    OPTIONS_STU op;
    parse_arg(argc, argv, &op);
    if ((op.options & OPT_ERROR) == 0) {
        if (op.options & OPT_MILISEC) {
            milisec = 1;
            bmilisec->value(milisec);
        }
        if (op.options & OPT_COURSE) {
            heading = 1;
            bheading->value(heading);
        }
        if (op.options & OPT_TIME) {
            tmoff = op.tmoff;
            char bf[20];
            sprintf(bf, "%d", tmoff);
            intime->value(bf);
        }
        if (op.options & OPT_SOURCE) {
            tx_inp->value(op.infile);
            indir = fl_filename_isdir(op.infile)? 1: 0;
            bindir->value(indir);
        }
        if (op.options & OPT_DESTINATION) {
            tx_out->value(op.outdir);
        }
    }

    display_tx("\n");
    display_tx("Provide input data and press [Convert].\n");
    display_tx("\n");
    display_tx("You can provide nuvi logs directory as an input,\n");
    display_tx("wNutrak will convert all files gps.bin from logs subdirectories.\n");
    display_tx("\n");
    tx_win->show_insert_position();

    window->show(1, argv);

    Fl::run();

    return (0);
}

